<?php
// This file handles the logic for a user paying with their wallet balance.
require_once 'includes/init.php'; // Uses the central logic file

// Security: Ensure a user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}
$user_id = $_SESSION['user_id'];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $booking_ref = $_POST['booking_ref'] ?? '';

    if (empty($booking_ref)) {
        die('Invalid request.');
    }

    // Use a transaction for this financial operation
    $conn->begin_transaction();
    try {
        // 1. Get the booking details AND lock the row for the update
        $stmt_booking = $conn->prepare("SELECT user_id, total_price, payment_status FROM bookings WHERE booking_ref = ? FOR UPDATE");
        $stmt_booking->bind_param("s", $booking_ref);
        $stmt_booking->execute();
        $booking = $stmt_booking->get_result()->fetch_assoc();
        $stmt_booking->close();

        // 2. Get the user's wallet balance AND lock the row
        $stmt_user = $conn->prepare("SELECT wallet_balance FROM users WHERE id = ? FOR UPDATE");
        $stmt_user->bind_param("i", $user_id);
        $stmt_user->execute();
        $user = $stmt_user->get_result()->fetch_assoc();
        $stmt_user->close();

        // 3. Perform crucial validations
        if (!$booking || $booking['user_id'] != $user_id) {
            throw new Exception("Booking not found or does not belong to you.");
        }
        if ($booking['payment_status'] === 'Paid') {
            throw new Exception("This booking has already been paid for.");
        }
        if ($user['wallet_balance'] < $booking['total_price']) {
            throw new Exception("Insufficient wallet balance.");
        }

        // 4. All checks passed, proceed with the transaction
        $new_balance = $user['wallet_balance'] - $booking['total_price'];

        // a) Debit the user's wallet
        $stmt_debit = $conn->prepare("UPDATE users SET wallet_balance = ? WHERE id = ?");
        $stmt_debit->bind_param("di", $new_balance, $user_id);
        $stmt_debit->execute();
        $stmt_debit->close();

        // b) Mark the booking as paid
        $stmt_paid = $conn->prepare("UPDATE bookings SET payment_status = 'Paid' WHERE booking_ref = ?");
        $stmt_paid->bind_param("s", $booking_ref);
        $stmt_paid->execute();
        $stmt_paid->close();

        // c) Record the transaction
        $description = "Payment for booking " . $booking_ref;
        $stmt_trans = $conn->prepare("INSERT INTO wallet_transactions (user_id, type, amount, description, reference_id) VALUES (?, 'Payment', ?, ?, ?)");
        $stmt_trans->bind_param("idss", $user_id, $booking['total_price'], $description, $booking_ref);
        $stmt_trans->execute();
        $stmt_trans->close();

        // 5. If everything is successful, commit the changes
        $conn->commit();

        // Redirect to a success page
        header("Location: payment-success.php?ref=" . $booking_ref);
        exit();

    } catch (Exception $e) {
        // If anything fails, roll back all changes
        $conn->rollback();
        // You can set an error message in the session and redirect back
        $_SESSION['error_message'] = $e->getMessage();
        header("Location: confirmation.php?booking_ref=" . $booking_ref);
        exit();
    }
} else {
    // If not a POST request, redirect away
    header('Location: index.php');
    exit();
}
