<?php 
// Set the page title before including the header
$page_title = 'Hotel Details'; // This will be updated after fetching the hotel name

// This should be the first file included. It handles the session, DB connection, and <head> section.
require_once 'includes/header.php'; 

// --- 1. GET AND VALIDATE HOTEL ID ---
$hotel_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$hotel_id) {
    // A simple way to handle invalid ID, you can make this a nicer "Not Found" page
    echo "<div class='container my-5'><div class='alert alert-danger'>Invalid hotel ID provided.</div></div>";
    require_once 'includes/footer.php';
    exit();
}

// --- 2. FETCH HOTEL DATA FROM DATABASE ---
// Fetch main hotel details
$stmt = $conn->prepare("SELECT * FROM hotels WHERE id = ? AND is_published = 1");
$stmt->bind_param("i", $hotel_id);
$stmt->execute();
$result = $stmt->get_result();
$hotel = $result->fetch_assoc();
$stmt->close();

// If hotel doesn't exist or isn't published, show an error
if (!$hotel) {
    echo "<div class='container my-5'><div class='alert alert-warning'>Sorry, the requested hotel was not found.</div></div>";
    require_once 'includes/footer.php';
    exit();
}

// Now we have the hotel name, let's update the page title
$page_title = htmlspecialchars($hotel['name']);

// Fetch gallery images for this hotel
$stmt = $conn->prepare("SELECT image_url FROM galleries WHERE item_id = ? AND item_type = 'hotel' ORDER BY display_order ASC");
$stmt->bind_param("i", $hotel_id);
$stmt->execute();
$gallery_result = $stmt->get_result();
$gallery_images = $gallery_result->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch rooms and their base rates for this hotel
$stmt = $conn->prepare("
    SELECT r.id, r.room_type_name, r.description, r.max_occupancy, MIN(rr.price_per_night) as starting_price
    FROM hotel_rooms r
    LEFT JOIN hotel_room_rates rr ON r.id = rr.room_id AND rr.is_active = 1
    WHERE r.hotel_id = ?
    GROUP BY r.id
    ORDER BY starting_price ASC
");
$stmt->bind_param("i", $hotel_id);
$stmt->execute();
$rooms_result = $stmt->get_result();
$rooms = $rooms_result->fetch_all(MYSQLI_ASSOC);
$stmt->close();
?>

<div class="container my-4">
    <div class="row">
        <div class="col-lg-8">
            <div class="mb-3">
                <h1 class="display-5 fw-bold"><?php echo htmlspecialchars($hotel['name']); ?></h1>
                <div class="d-flex align-items-center text-muted">
                    <div class="me-3">
                        <?php for ($i = 0; $i < $hotel['star_rating']; $i++): ?>
                            <i class="bi bi-star-fill text-warning"></i>
                        <?php endfor; ?>
                    </div>
                    <div>
                        <i class="bi bi-geo-alt-fill"></i> <?php echo htmlspecialchars($hotel['location']); ?>
                    </div>
                </div>
            </div>

            <div class="mb-4">
                <img src="<?php echo htmlspecialchars($hotel['main_image_url']); ?>" class="img-fluid rounded shadow-sm w-100" alt="Main image of <?php echo htmlspecialchars($hotel['name']); ?>">
                <?php if (!empty($gallery_images)): ?>
                <div class="row g-2 mt-2">
                    <?php foreach (array_slice($gallery_images, 0, 4) as $image): // Show up to 4 thumbnails ?>
                    <div class="col-3">
                        <img src="<?php echo htmlspecialchars($image['image_url']); ?>" class="img-fluid rounded" alt="Gallery image">
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>

            <div class="card shadow-sm mb-4">
                <div class="card-header bg-light">
                    <h5 class="mb-0">About this Hotel</h5>
                </div>
                <div class="card-body">
                    <?php
                        // THE FIX: Directly echo the description to render HTML tags.
                        // This assumes the content from the admin's rich text editor is trusted.
                        echo $hotel['description']; 
                    ?>
                </div>
            </div>

            <?php if (!empty($hotel['amenities'])): ?>
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-light"><h5 class="mb-0">Amenities</h5></div>
                <div class="card-body">
                    <div class="row">
                        <?php 
                            $amenities = explode(',', $hotel['amenities']);
                            foreach ($amenities as $amenity): 
                        ?>
                        <div class="col-md-6 col-lg-4 mb-2">
                            <i class="bi bi-check-circle-fill text-success"></i> <?php echo htmlspecialchars(trim($amenity)); ?>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>

             <div class="card shadow-sm mb-4">
                <div class="card-header bg-light"><h5 class="mb-0">Available Rooms</h5></div>
                <div class="list-group list-group-flush">
                    <?php foreach ($rooms as $room): ?>
                    <div class="list-group-item p-3">
                        <div class="d-flex w-100 justify-content-between">
                            <h6 class="mb-1 fw-bold"><?php echo htmlspecialchars($room['room_type_name']); ?></h6>
                            <span class="text-primary fs-5 fw-bold">
                                <?php if($room['starting_price']): ?>
                                    $<?php echo number_format($room['starting_price'], 2); ?>
                                    <small class="text-muted fw-normal">/night</small>
                                <?php else: ?>
                                    <small class="text-muted fw-normal">Price on request</small>
                                <?php endif; ?>
                            </span>
                        </div>
                        <p class="mb-1"><small>Max Occupancy: <?php echo htmlspecialchars($room['max_occupancy']); ?> person(s)</small></p>
                        <a href="#booking-widget" class="btn btn-sm btn-primary mt-2">Book Now</a>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>

        </div>

        <div class="col-lg-4">
            <div class="card shadow-sm sticky-top" style="top: 100px;" id="booking-widget">
                <div class="card-body p-4">
                    <h4 class="card-title text-center mb-4">Check Availability</h4>
                    <form>
                        <div class="mb-3">
                            <label for="checkin" class="form-label">Check-in Date</label>
                            <input type="date" class="form-control" id="checkin" name="checkin">
                        </div>
                        <div class="mb-3">
                            <label for="checkout" class="form-label">Check-out Date</label>
                            <input type="date" class="form-control" id="checkout" name="checkout">
                        </div>
                         <div class="mb-3">
                            <label for="guests" class="form-label">Guests</label>
                            <input type="number" class="form-control" id="guests" name="guests" value="2" min="1">
                        </div>
                        <div class="d-grid">
                            <button type="submit" class="btn btn-primary btn-lg">Check Availability</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>


<?php 
// This should be the last file included.
require_once 'includes/footer.php'; 
?>
