<?php
require_once 'header.php'; 
$page_title = 'My Ferry Bookings';

// --- PAGE-SPECIFIC QUERY ---
// Fetches all ferry bookings for the logged-in user, joining with the ports table to get names.
$stmt = $conn->prepare("
    SELECT 
        b.id,
        b.booking_ref,
        b.departure_date,
        b.num_passengers,
        b.status,
        orig.name AS origin_name,      -- CORRECTED: Was 'orig.city'
        dest.name AS destination_name  -- CORRECTED: Was 'dest.city'
    FROM 
        ferry_bookings b
    JOIN 
        ferry_routes r ON b.route_id = r.id
    JOIN 
        ferry_ports orig ON r.origin_port_id = orig.id
    JOIN 
        ferry_ports dest ON r.destination_port_id = dest.id
    WHERE 
        b.user_id = ?
    ORDER BY 
        b.departure_date DESC
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$bookings = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Helper to get a Bootstrap class based on the booking status
function get_status_badge($status) {
    switch (strtolower($status)) {
        case 'confirmed': return 'bg-success';
        case 'pending': return 'bg-warning text-dark';
        case 'quoted': return 'bg-info';
        case 'cancelled': return 'bg-danger';
        default: return 'bg-secondary';
    }
}
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2"><i class="bi bi-tsunami me-2"></i>My Ferry Bookings</h1>
    <a href="../ferries.php" class="btn btn-primary"><i class="bi bi-plus-circle-fill me-1"></i> Book a New Ferry</a>
</div>

<div class="card shadow-sm border-0">
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-hover align-middle">
                <thead class="table-light">
                    <tr>
                        <th>Reference</th>
                        <th>Route</th>
                        <th>Departure Date</th>
                        <th class="text-center">Passengers</th>
                        <th class="text-center">Status</th>
                        <th class="text-end">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($bookings)): ?>
                        <tr>
                            <td colspan="6" class="text-center p-5">
                                <i class="bi bi-search fs-2 text-muted"></i>
                                <p class="mt-2 mb-0 text-muted">You have no ferry bookings yet.</p>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($bookings as $booking): ?>
                        <tr>
                            <td class="fw-bold"><?php echo htmlspecialchars($booking['booking_ref'] ?? 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($booking['origin_name']); ?> &rarr; <?php echo htmlspecialchars($booking['destination_name']); ?></td>
                            <td><?php echo date('F j, Y', strtotime($booking['departure_date'])); ?></td>
                            <td class="text-center"><?php echo htmlspecialchars($booking['num_passengers']); ?></td>
                            <td class="text-center">
                                <span class="badge <?php echo get_status_badge($booking['status']); ?>">
                                    <?php echo htmlspecialchars(ucfirst($booking['status'])); ?>
                                </span>
                            </td>
                            <td class="text-end">
                                <a href="view-ferry-booking.php?id=<?php echo $booking['id']; ?>" class="btn btn-sm btn-primary">View Details</a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<?php 
require_once 'footer.php'; 
?>
