<?php
require_once '../includes/db.php';

// Security check: Ensure a user is logged in and their role is 'customer'.
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'customer') {
    // If they are another role (like admin), redirect them to their correct dashboard.
    if (isset($_SESSION['user_role']) && $_SESSION['user_role'] == 'admin') {
        header("Location: ../admin/dashboard.php");
    } elseif (isset($_SESSION['user_role']) && in_array($_SESSION['user_role'], ['tour_partner', 'hotel_partner'])) {
        header("Location: ../partner/");
    } else {
        // Otherwise, send them to the login page.
        header("Location: ../login.php");
    }
    exit();
}

$user_id = $_SESSION['user_id'];
// Fetch the full, current user data on every page load for consistency and security
$stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$user = $stmt->get_result()->fetch_assoc();

if (!$user) {
    // If the user ID in the session doesn't exist in the DB, destroy the session and log them out.
    session_destroy();
    header("Location: ../login.php?error=invalid_session");
    exit();
}

$wallet_balance = $user['wallet_balance'];
$_SESSION['user_name'] = $user['full_name']; // Keep session name in sync

// Fetch unread message count for the notification badge
$unread_messages_count = 0;
$stmt_msg = $conn->prepare("SELECT COUNT(*) as total FROM messages WHERE recipient_id = ? AND is_read = 0");
$stmt_msg->bind_param("i", $user_id);
$stmt_msg->execute();
$unread_messages_count = $stmt_msg->get_result()->fetch_assoc()['total'];

// Fetch global site settings
$settings_result = $conn->query("SELECT * FROM site_settings");
$site_settings = [];
if ($settings_result) {
    while ($row = $settings_result->fetch_assoc()) {
        $site_settings[$row['setting_name']] = $row['setting_value'];
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Dashboard - <?php echo htmlspecialchars($site_settings['site_name'] ?? ''); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
    <link rel="stylesheet" href="/css/style.css">
    
    <?php if (isset($site_settings['paypal_enabled']) && $site_settings['paypal_enabled'] == '1' && !empty($site_settings['paypal_client_id'])): ?>
        <script src="https://www.paypal.com/sdk/js?client-id=<?php echo htmlspecialchars($site_settings['paypal_client_id']); ?>&currency=<?php echo htmlspecialchars($site_settings['paypal_currency'] ?? 'PHP'); ?>"></script>
    <?php endif; ?>
</head>
<body>
<nav class="navbar navbar-expand-lg navbar-dark bg-dark sticky-top shadow-sm">
  <div class="container-fluid">
    <a class="navbar-brand d-flex align-items-center" href="../index.php">
        <?php if (!empty($site_settings['site_logo_url'])): ?>
            <img src="../<?php echo htmlspecialchars($site_settings['site_logo_url']); ?>" alt="Logo" style="max-height: 40px; margin-right: 10px;">
        <?php else: ?>
            <i class="bi bi-compass-fill me-2"></i> <?php echo htmlspecialchars($site_settings['site_name'] ?? 'DXP Tours'); ?>
        <?php endif; ?>
    </a>
    <div class="dropdown ms-auto">
        <a href="#" class="d-flex align-items-center text-white text-decoration-none dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">
            <?php if (!empty($user['profile_image_url'])): ?>
                <img src="../<?php echo htmlspecialchars($user['profile_image_url']); ?>" alt="Profile" width="32" height="32" class="rounded-circle me-2" style="object-fit: cover;">
            <?php else: ?>
                <div class="avatar-initial me-2"><?php echo strtoupper(substr($user['full_name'], 0, 1)); ?></div>
            <?php endif; ?>
            <strong><?php echo htmlspecialchars($user['full_name']); ?></strong>
        </a>
        <ul class="dropdown-menu dropdown-menu-dark dropdown-menu-end text-small shadow">
            <li><a class="dropdown-item" href="edit-profile.php"><i class="bi bi-person-fill-gear me-2"></i> Edit Profile</a></li>
            <li><hr class="dropdown-divider"></li>
            <li><a class="dropdown-item" href="../logout.php"><i class="bi bi-box-arrow-right me-2"></i> Log Out</a></li>
        </ul>
    </div>
  </div>
</nav>

<div class="container-fluid">
    <div class="row">
        <nav class="col-md-3 col-lg-2 d-md-block bg-light sidebar collapse vh-100 border-end">
            <div class="d-flex flex-column h-100 pt-3">
                <ul class="nav flex-column">
                    <li class="nav-item"><a class="nav-link" href="index.php"><i class="bi bi-grid-1x2-fill me-2"></i> Dashboard</a></li>
                    <li class="nav-item"><a class="nav-link d-flex justify-content-between align-items-center" href="view-messages.php"><i class="bi bi-chat-dots-fill me-2"></i> Messages <?php if($unread_messages_count > 0) echo '<span class="badge bg-danger rounded-pill">'.$unread_messages_count.'</span>'; ?></a></li>
                    
                    <li class="nav-item mt-2">
                        <a class="nav-link d-flex justify-content-between align-items-center" data-bs-toggle="collapse" href="#bookingsMenu">
                            <span><i class="bi bi-briefcase-fill me-2"></i> My Bookings</span><i class="bi bi-chevron-down small"></i>
                        </a>
                        <div class="collapse" id="bookingsMenu">
                            <ul class="nav flex-column ps-4">
                                <li class="nav-item"><a class="nav-link py-1" href="my-tour-bookings.php"><i class="bi bi-card-list me-2"></i> Tours & Services</a></li>
                                <li class="nav-item"><a class="nav-link py-1" href="my-hotel-bookings.php"><i class="bi bi-building me-2"></i> Hotels</a></li>
                                <li class="nav-item"><a class="nav-link py-1" href="my-flight-bookings.php"><i class="bi bi-airplane me-2"></i> Flights</a></li>
                                <li class="nav-item"><a class="nav-link py-1" href="my-ferry-bookings.php"><i class="bi bi-tsunami me-2"></i> Ferries</a></li>
                            </ul>
                        </div>
                    </li>

                    <li class="nav-item mt-2">
                        <a class="nav-link d-flex justify-content-between align-items-center" data-bs-toggle="collapse" href="#accountMenu">
                           <span><i class="bi bi-person-vcard-fill me-2"></i> My Account</span><i class="bi bi-chevron-down small"></i>
                        </a>
                        <div class="collapse" id="accountMenu">
                            <ul class="nav flex-column ps-4">
                                <li class="nav-item"><a class="nav-link py-1" href="edit-profile.php"><i class="bi bi-person-fill-gear me-2"></i> Edit Profile</a></li>
                                <li class="nav-item"><a class="nav-link py-1" href="my-reviews.php"><i class="bi bi-star-fill me-2"></i> My Reviews</a></li>
                                <li class="nav-item"><a class="nav-link py-1" href="affiliate.php"><i class="bi bi-people-fill me-2"></i> Affiliate Program</a></li>
                                <li class="nav-item"><a class="nav-link py-1" href="top-up.php"><i class="bi bi-wallet2 me-2"></i> Top-Up Wallet</a></li>
                                <li class="nav-item"><a class="nav-link py-1" href="transactions.php"><i class="bi bi-clock-history me-2"></i> Transactions</a></li>
                            </ul>
                        </div>
                    </li>
                </ul>
                <ul class="nav flex-column mt-auto mb-2">
                     <li class="nav-item"><a class="nav-link text-danger" href="../logout.php"><i class="bi bi-box-arrow-right me-2"></i> Log Out</a></li>
                </ul>
            </div>
        </nav>
        <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 bg-light">
